

#include "externs.h"

/*
 * Process data in a line status register
 */
static int
process_lsr(DEV_8250 *dev, unsigned char lsr) {
	unsigned key = 0, eventflag = 0;

	// Return immediately if no errors.
	if((lsr & (LSR_BI|LSR_OE|LSR_FE|LSR_PE)) == 0)
		return(0);

	// Save the error as out-of-band data which can be retrieved via devctl().
	dev->tty.oband_data |= (lsr >> 1) & 0x0f;
	atomic_set(&dev->tty.flags, OBAND_DATA);

	// Read whatever input data happens to be in the buffer to "eat" the
	// spurious data associated with break, parity error, etc.
	key = read_8250(dev->port[REG_RX]);

	if(lsr & LSR_BI)
		key |= TTI_BREAK;
	else if(lsr & LSR_OE)
		key |= TTI_OVERRUN;
	else if(lsr & LSR_FE)
		key |= TTI_FRAME;
	else if(lsr & LSR_PE)
		key |= TTI_PARITY;

	return(tti(&dev->tty, key) | eventflag);
	}

/*
 * Serial interrupt handler
 */
const struct sigevent *
ser_intr(void *area, int id) {
	struct dev_list	*list = area;
	int				status = 0;
	int				something_happened;
	unsigned char	msr, lsr;
	DEV_8250		*dev;
	struct sigevent *event = NULL;
	unsigned   fifo_counter, bytes_in_fifo;

	do {
		something_happened = 0;

		for(dev = list->device; dev != NULL; dev = dev->next) {
			unsigned	iir;
			uintptr_t	*port = dev->port;

			status = 0;

			iir = read_8250(port[REG_II]) & 0x07;
			switch(iir) {
			case II_RX:		// Receive data
				fifo_counter = read_8250(port[XR_RXFIFOTRIG_REGISTER]);
				while(fifo_counter)
				{
					status |= tti(&dev->tty, read_8250(port[REG_RX]));
					fifo_counter--;
				}
				break;

			case II_TX:		// Transmit buffer empty
				if(dev->dev_id > 0x258) //PCIe
				{
					bytes_in_fifo = read_8250(port[XR_TXFIFOTRIG_REGISTER]);				
					fifo_counter = 256 - bytes_in_fifo;
				}
				else
				{
					bytes_in_fifo = read_8250(port[XR_TXFIFOTRIG_REGISTER]) % 64;
					fifo_counter = 64 - bytes_in_fifo;
				}

				do {
					dev->tty.un.s.tx_tmr = 0;
					// The override causes tto to ignore the check on tx buffer empty 
					// since we know that the entire fifo is empty when the interrupt fires
					//status |= tto(&dev->tty, TTO_DATA, dev->fifo_override);
					status |= tto(&dev->tty, TTO_DATA, FIFO_XMIT_OVERRIDE);
				} while( (--fifo_counter) && (dev->tty.obuf.cnt > 0) );
				break;

			case II_MS:		// Modem change
				msr = read_8250(port[REG_MS]);

				if(msr & MSR_DDCD)
					status |= tti(&dev->tty, (msr & MSR_DCD) ? TTI_CARRIER : TTI_HANGUP);
					
				if((msr & MSR_DCTS)  &&  (dev->tty.c_cflag & OHFLOW))
					status |= tti(&dev->tty, (msr & MSR_CTS) ? TTI_OHW_CONT : TTI_OHW_STOP);
				break;

			case II_LS:		// Line status change
				lsr = read_8250(port[REG_LS]);
				status |= process_lsr(dev, lsr);
				break;

			default:
				continue;

				}

			something_happened = 1;
			if(status) {
				if((dev->tty.flags & EVENT_QUEUED) == 0) {
					event = &ttyctrl.event;
					dev_lock(&ttyctrl);
					ttyctrl.event_queue[ttyctrl.num_events++] = &dev->tty;
					atomic_set(&dev->tty.flags, EVENT_QUEUED);
					dev_unlock(&ttyctrl);
					}
				}
			}

		} while(something_happened);

	return(event);
	}
